	function [b, c] = iirNegGrp(n, tau, t1, om0, apptyp)
	
	%	iirNegGrp: Design IIR filter with negative group delay
	%
	%   Syntax:
	%   [b,c]=iirNegGrp(n,tau,'maxflat')
	%   [b,c]=iirNegGrp(n,tau,t1,'maxflat')
	%   [b,c]=iirNegGrp(n,tau,t1,om0)
	%   [b,c]=iirNegGrp(n,tau,t1,om0,'Tscheby')
	%   [b,c]=iirNegGrp(n,tau,[],om0,'Tscheby')
	%
	%   Description: for parameters, see [DSV2, page 490]
	%       t1:         default: t1=n-0.5
	%       apptyp:     default: 'Tscheby'
	%
	%   Algorithm: 
	%
	%   Remark:
	%
	%   References:[DSV2, chapt. 4.5.2]
	%
	%   See also  DSV-Lib: allTslinpha, allGrpMaxflat
	%             Matlab:
	
	%% ---------------------------------------------------------------
	%   DSV2-Bibliothek zu H.W. Schuessler: Digitale Signalverarbeitung 2,
	%                      Springer (2010)
	%   Authors: Hans W. Schuessler and Guenter F. Dehner
	%   Version: 1.0
	%   Copyright 2009 by authors - not released for commercial use
	%%  ---------------------------------------------------------------
%				error(nargchk(3,5,nargin, 'struct'))
	%
	if nargin <5, apptyp='Tscheby'; end
	if ischar(t1),  apptyp=t1;   t1=[]; om0=[];   end
	if ischar(om0), apptyp=om0;  om0=[];   end
	if isempty(t1), t1 = n-0.5;  end
	t2 = t1 - 2*tau;
	switch lower(apptyp)
	    case 'tscheby'
	        if isempty(om0), error('firNegGrp: Parameter missing!'); end
	        b = allTslinpha(n,t2,om0);
	        c=allTslinpha(n,t1,om0);
	    case 'maxflat'
	        b=allGrpMaxflat(n,t2);
	        c=allGrpMaxflat(n,t1);
	    otherwise, error('firNegGRP: Type not defined')
	end
	%% [EOF] - iirNegGrp

		function c = allGrpMaxflat(n,t0)
		
	%allGrp_maxflat: Approximation of allpass with maximal flat groupdelay
	%
	%   Syntax:
	%   c = allGrpMaxflat(n,t0)
	%
	%   Description: [DSV2, page 255]
	% c = flachtg(n,t0) calculates the denominator polynomial of an order n 
	%   allpass with maximally flat groupdelay. The allpass shows for om=0 the
	%   groupdelay t0.
	%
	%   Algorithm: 
	%
	%   Remark:
	%
	%   References: [DSV2, chap. 3.6.3]
	%
	%   See also  DSV-Lib: 
	%             Matlab: cumprod
	
	%% ---------------------------------------------------------------
	%   DSV2-Bibliothek zu H.W. Schuessler: Digitale Signalverarbeitung 2,
	%                      Springer (2010)
	%   Authors: Hans W. Schuessler and Guenter F. Dehner
	%   Version: 1.0        Release DSV 2010/1
	%   Copyright 2009 by authors - not released for commercial use
	%%  ---------------------------------------------------------------
	error(nargchk(3,3,nargin, 'struct'))
	%
	nu = 0:n-1;
	c = cumprod([1, (nu-n).*(t0+nu-n)./(nu+1)./(t0+1+nu)]);
	%% [EOF] - allGrpMaxflat

		function [c,c0,Delta] = allTslinpha(n,t0,omg,tol)
	%	allTslinpha: Chebyshev approximation od an allpass with linear phase
	%
	%   Syntax:
	%   c = allTslinpha(n,t0,omg)
	%   c = allTslinpha(n,t0,omg,tol)
	%   [c,c0,Delta] = allTslinpha(...)
	%
	%   Description: [DSV2, page 250]
	% c = allTslinpha(n,t0,omg) calculates the denominator c for the
	%   approximated linear-phase allpass of order n and the desired delay t0 
	%   in the bounds om=[0 omg]. The tolerance for termination of the
	%   optimization process is set to tol = 1e-10 by default. To avoid early
	%   termination a modification of the limit tol may be successful.  
	%
	% [c,c0,Delta] = allTslinpha(...) returns in addition the start solution c0
	%   and the resulting normalized deviation of the phase Delta = max(db(om))/pi.
	%
	%   Algorithm: Chebyshev Approximation 
	%
	%   Remark:
	%
	%   References: [DSV2, chap. 3.6.3], [La92]
	%
	%   See also  DSV-Lib: apphase, loc_max
	%             Matlab:  median, mldivide, matrix left division
	
	%% ---------------------------------------------------------------
	%   DSV2-Bibliothek zu H.W. Schuessler: Digitale Signalverarbeitung 2,
	%                      Springer (2010)
	%   Authors: Hans W. Schuessler and Guenter F. Dehner
	%   Version: 1.0        Release DSV 2010/1
	%   Copyright 2009 by authors - not released for commercial use
	%%  ---------------------------------------------------------------
%	error(nargchk(3,4,nargin, 'struct'))
	%
	if nargin < 4, tol = 1e-10; end
	N  = 1000;
	om = (0:N)/N;                                
	omi = (1:fix(omg*100))/100; omgind = round(omg*N);
	omi = omi(:)*pi; bwi = omi*t0;
	betai = .5*(n*omi + bwi);
	%
	A = sin(omi*(n-1:-1:0)-betai*ones(1,n));              % start solution
	b = -sin(n*omi-betai);
	c_ = A\b; c = [1;c_]; c0 = c';
	
	% Remez-Algorithm
	ii=0; dbmax = realmax;
	while dbmax > tol                                     % determination
	   ii=ii+1;
	   if ii>100,
	      warning('DSVLib:Conv',' No convergence in allTslinpha');  break
	   end
	   ba = apphase(c,om); db = ba -t0*om;                % extrema
	   omax = loc_max(db); omin = loc_max(-db);
	   omex = sort([omax;omin]); omex = [omex(2:n+1);omgind];
	   d0 = median(abs(db(omex)));
	   dbmax = max(abs(db(omex)))-d0;
	   omi = [om(omex(1:n))';omg]*pi;
	   i = 1:n+1; d = d0*pi; Delta = d0;
	   jj=0; dx=realmax;  
	   while abs(dx) >  tol,                                % interpolation
	      jj=jj+1;
	      if jj>100,
	         warning('DSVLib:Conv',' No convergence in allTslinpha'); break
	      end
	      x = [c(2:n+1);d];                                  % solution by
	      betai = .5*((t0+n)*omi - d*(-1).^i');              % Newton-Raphson     
	      f = sin(omi*(n:-1:0)-betai*ones(1,n+1))*c;
	      J = [sin(omi*(n-1:-1:0)-betai*ones(1,n)),...
	          .5*diag((-1).^i)*cos(omi*(n:-1:0)-betai*ones(1,n+1))*c];
	      dx = J\f; 
	      x  = x-dx; c = [1; x(1:n)]; d = x(n+1);
	   end
	end 
	c=c';
	%% [EOF] - allTslinpha
	
		function [ph,W] = apphase(c,om,fs)
	%	apphase: Phase response of an allpass
	%
	%   Syntax:
	%   [ph,W] = apphase(c)
	%   [ph,W] = apphase(c,N)
	%   [ph,W] = apphase(c,om);    om = [0, 1)
	%   [ph,W] = apphase(c,f,fs)    f = [0, fs/2)
	%
	%   Description: 
	%  ph=apphase(c,om) calculates the phase response of an
	%   allpass, normalized to 'pi' for the input frequency points om=omega/pi.
	%   The allpass is defined by the denominator polynom of its transfer
	%   function. The phase response is represented in its unwrapped form.
	%  ph=apphase(c,N) calculates the phase response for N frequencies equally
	%   distributed in the range om=[0,1). By default N=512.
	%
	%   Algorithm: 
	%
	%   Remark: The vector om should be not too short, so that unwrapping makes
	%           sense.   
	%
	%   References: References:[DSV2, chapt. 3.7.2]
	%
	%   See also  DSV-Lib: apgrpdelay
	%             Matlab:  freqz, zerophase
	
	%% ---------------------------------------------------------------
	%   DSV2-Bibliothek zu H.W. Schuessler: Digitale Signalverarbeitung 2,
	%                      Springer (2010)
	%   Authors: Hans W. Schuessler and Guenter F. Dehner
	%   Version: 1.0        Release DSV 2010/1
	%   Copyright 2009 by authors - not released for commercial use
	%%  ---------------------------------------------------------------
%	error(nargchk(1,3,nargin, 'struct'))
	%
	if nargin < 3, fs=1; end
	if nargin < 2, N=512; om = (0:N-1)/N;     
	else 
	    N=length(om);
	    if N==1, N=om;  om = (0:N-1)/N; end
	end
	om=om/fs;
	%%   
	r=size(om,1);
	om = om(:); c = c(:); n = length(c)-1; nue = n:-1:0;
	ph = unwrap(2*angle(exp(1i*om*pi*nue)*c))/pi - n*om;
	W = om*pi;
	if r==1, ph=ph(:)'; W=W'; end
	%% [EOF] - apphase

		function ind = loc_max(y)
	%loc_max: Determination of the local extremas within a data sequence
	%
	%   Syntax:
	%   ind = loc_max(y)
	%
	%   Description: see [DSV2, p. 16]
	% ind = loc_max(y) determines the indices of the local maxima of data
	%   the vector y
	%
	%   Algorithm: 
	%
	%   Remark:
	%
	%   References: [DSV2, chap. 2.3.1]
	%
	%   See also  DSV-Lib: 
	%             Matlab:  sort, find
	
	%% ---------------------------------------------------------------
	%   DSV2-Bibliothek zu H.W. Schuessler: Digitale Signalverarbeitung 2,
	%                      Springer (2010)
	%   Authors: Hans W. Schuessler and Guenter F. Dehner
	%   Version: 1.0        Release DSV 2010/1
	%   Copyright 2009 by authors - not released for commercial use
	%%  ---------------------------------------------------------------
%	error(nargchk(1,1,nargin, 'struct'))
	%
	L = length(y) + 2;
	y = [y(1)-1; y(:); y(L-2)-1];
	kl = y(1:L-1) < y(2:L); 
	gr = y(1:L-1) > y(2:L);
	ind = sort(find(kl(1:L-2) & gr(2:L-1)));
	ind = ind(:);
	%% [EOF] - loc_max

	
